﻿using System;

namespace VA.PPMS.IWS.Common
{
    public class ShortGuid
    {
        private readonly Guid guid;
        private readonly string value;

        /// <summary>Create a 22-character case-sensitive short GUID.</summary>
        public ShortGuid(Guid guid)
        {
            if (guid == null)
            {
                throw new ArgumentNullException("guid");
            }

            this.guid = guid;
            this.value = Convert.ToBase64String(guid.ToByteArray())
                .Substring(0, 22)
                .Replace("/", "_")
                .Replace("+", "-");
        }

        /// <summary>Get the short GUID as a string.</summary>
        public override string ToString()
        {
            return this.value;
        }

        public string ToBase64()
        {
            return Convert.ToBase64String(System.Text.Encoding.UTF8.GetBytes(this.value));
        }

        /// <summary>Get the Guid object from which the short GUID was created.</summary>
        public Guid ToGuid()
        {
            return this.guid;
        }

        public static ShortGuid ParseBase64(string shortGuid)
        {
            var guidStr = System.Text.Encoding.UTF8.GetString(Convert.FromBase64String(shortGuid));
            return Parse(guidStr);
        }

        /// <summary>Get a short GUID as a Guid object.</summary>
        /// <exception cref="System.ArgumentNullException"></exception>
        /// <exception cref="System.FormatException"></exception>
        public static ShortGuid Parse(string shortGuid)
        {
            if (shortGuid == null)
            {
                throw new ArgumentNullException("shortGuid");
            }
            else if (shortGuid.Length != 22)
            {
                throw new FormatException("Input string was not in a correct format.");
            }

            return new ShortGuid(new Guid(Convert.FromBase64String
                (shortGuid.Replace("_", "/").Replace("-", "+") + "==")));
        }

        public static implicit operator String(ShortGuid guid)
        {
            return guid.ToString();
        }

        public static implicit operator Guid(ShortGuid shortGuid)
        {
            return shortGuid.guid;
        }
    }
}
